<?php
/* --------------------------------------------------------------
 ApiServiceProviderRegistration.php 2021-05-14
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2020 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

declare(strict_types=1);

namespace Gambio\Api\Application\Bootstrapper;

use Gambio\Api\Application\Auth\RequestAuthenticationServiceServiceProvider;
use Gambio\Api\Application\ErrorHandling\ApiErrorHandlerServiceProvider;
use Gambio\Api\Application\Middleware\ApiMiddlewareServiceProvider;
use Gambio\Core\Application\Application;
use Gambio\Core\Application\Bootstrapper;
use Gambio\Core\Application\ServiceProviderRegistry\ServiceProviderRegistry;
use Gambio\Core\Application\ServiceProviderRegistry\ServiceProviderRegistryCache;
use Gambio\Core\Application\ServiceProviderRegistry\ServiceProviderRegistryType;
use Gambio\Core\Application\ValueObjects\Environment;
use Gambio\Core\Cache\Services\CacheFactory;
use RuntimeException;

/**
 * Class ApiServiceProviderRegistration
 *
 * @package Gambio\Api\Application\Bootstrapper
 */
class ApiServiceProviderRegistration implements Bootstrapper
{
    private const API_COMPONENT_SERVICE_PROVIDERS = [
        RequestAuthenticationServiceServiceProvider::class,
        ApiMiddlewareServiceProvider::class,
        ApiErrorHandlerServiceProvider::class,
    ];
    
    
    /**
     * @inheritDoc
     */
    public function boot(Application $application): void
    {
        foreach (self::API_COMPONENT_SERVICE_PROVIDERS as $componentServiceProvider) {
            $application->registerProvider($componentServiceProvider);
        }
        
        $cacheFactory = $this->getCacheFactory($application);
        $environment  = $this->getEnvironment($application);
        
        $registryCache = new ServiceProviderRegistryCache($cacheFactory);
        $registry      = new ServiceProviderRegistry($registryCache, $environment);
        
        $this->registerModuleProvider($application, $registry, ServiceProviderRegistryType::api());
        $this->registerModuleProvider($application, $registry, ServiceProviderRegistryType::admin());
    }
    
    
    /**
     * Registers module service providers.
     *
     * @param Application                 $application
     * @param ServiceProviderRegistry     $registry
     * @param ServiceProviderRegistryType $type
     */
    public function registerModuleProvider(
        Application $application,
        ServiceProviderRegistry $registry,
        ServiceProviderRegistryType $type
    ): void {
        foreach ($registry->getRegistry($type) as $moduleProvider) {
            $application->registerProvider($moduleProvider);
        }
    }
    
    
    /**
     * @param Application $application
     *
     * @return CacheFactory
     */
    private function getCacheFactory(Application $application): CacheFactory
    {
        if (!$application->has(CacheFactory::class)) {
            throw new RuntimeException(CacheFactory::class . ' must be registered!');
        }
        
        return $application->get(CacheFactory::class);
    }
    
    
    /**
     * @param Application $application
     *
     * @return Environment
     */
    private function getEnvironment(Application $application): Environment
    {
        if (!$application->has(Environment::class)) {
            throw new RuntimeException(Environment::class . ' must be registered!');
        }
        
        return $application->get(Environment::class);
    }
}